#include <userint.h>
#include <formatio.h>
#include <ansi_c.h>
#include <utility.h>
#include <stdlib.h>
#include "gmmsp.h"
#include "gmmspu.h"

/*= Global Variable Definitions =============================================*/

long panelHandle[5];
unsigned long instrHandle;

/*= Message Arrays ==========================================================*/
static char *initCtrlsHelpStr[] = {
									"This control specifies the interface and address of the device\n"
									"that is to be initialized (Interface Descriptor). The exact\n"
									"grammar to be used in this control is shown in the note below.\n\n"
									"Default Value:  \"GPIB::11\"\n\n"


									"Interface   Grammar\n"
									"------------------------------------------------------\n"
									"GPIB        GPIB[board]::primary address[::secondary address]\n"
									"            [::INSTR]\n"
									"Serial      ASRL<port>::INSTR\n\n"
									"The ASRL keyword is used with Serial instruments.\n"
									"The GPIB keyword is used with GPIB instruments.\n",


                                    "This control specifies if an ID Query is sent to the\n"
                                    "instrument during the initialization procedure.\n\n"
                                    "Valid Range:\n"
                                    "0 - Skip Query\n"
                                    "1 - Do Query (Default Value)",
                                    
                                    
                                    "This control specifies if the instrument is to be reset to its\n"
                                    "power-on settings during the initialization procedure.\n\n"
                                    "Valid Range:\n"
                                    "0 - Don't Reset\n"
                                    "1 - Reset Device (Default Value)"};  
        
static char *configCtrlsHelpStr[] = {
    "This control returns output setpoint value\n" 
    "of the output current.",

                                    
    "This control returns output setpoint value\n" 
    "of the output voltage.",
                               
                                    
	"This control sets the setpoint value of the output current.\n\n"
	"Default Value: 0.1\n\n"
	"Valid Range:\n\n"
	"---------------------------------------------------------------------------------------------------\n"
	"|        Module             ch |  Valid Range (A)  |  Resolution (mA) |\n"
	"---------------------------------------------------------------------------------------------------\n"
	"|ES 31 K7 R7 P           A |       0...7.0 A       |          2 mA       |\n"
	"|ES 31 K7 R7 P           B |            -                |            -            |\n"
	"|ES 32 K30 R4 P         A |            -                |            -            |\n"
	"|ES 32 K30 R4 P         B |       0...4.0 A       |          1 mA       |\n"
	"|ES 32 K80 R1.5 P      A |            -                |            -            |\n"
	"|ES 32 K80 R1.5 P      B |       0...1.5 A       |       0.5 mA       |\n"
	"|ES 31 K2x8 R3 P       A |       0...3.0 A       |          1 mA       |\n"
	"|ES 31 K2x8 R3 P       B |       0...3.0 A       |          1 mA       |\n"
	"|ES 31 K2x16 R1.5 P  A |       0...1.5 A       |       0.5 mA       |\n"
	"|ES 31 K2x16 R1.5 P  B |       0...1.5 A       |       0.5 mA       |\n"
	"|ES 31 K2x40 R0.6 P  A |       0...0.6 A       |       0.2 mA       |\n"
	"|ES 31 K2x40 R0.6 P  B |       0...0.6 A       |       0.2 mA       |\n"
	"------------------------------------------------------------------------------------------------\n\n"
	"Note:\n"
	"(1) Value must be smaller than the current limit.\n",
	
	
	"This control sets the setpoint value of the output voltage.\n\n"
	"Default Value: 5.0\n\n"
	"Valid Range:\n"
	"-------------------------------------------------------------------------------------------------\n"
	"|           Module           ch |  Valid Range (V)  |  Resolution (mV) |\n"
	"-------------------------------------------------------------------------------------------------\n"
	"|ES 31 K7 R7 P           A |       0... 7 V           |           2 mV        |\n"
	"|ES 31 K7 R7 P           B |           -                 |              -            |\n"
	"|ES 32 K30 R4 P         A |           -                 |              -            |\n"
	"|ES 32 K30 R4 P         B |      0...30 V           |          10 mV       |\n"
	"|ES 32 K80 R1.5 P      A |           -                 |              -            |\n"
	"|ES 32 K80 R1.5 P      B |      0...80 V           |          20 mV       |\n"
	"|ES 31 K2x8 R3 P       A |       0... 8 V           |           2 mV        |\n"
	"|ES 31 K2x8 R3 P       B |       0... 8 V           |           2 mV        |\n"
	"|ES 31 K2x16 R1.5 P  A |      0...16 V           |           4 mV        |\n"
	"|ES 31 K2x16 R1.5 P  B |      0...16 V           |           4 mV        |\n"
	"|ES 31 K2x40 R0.6 P  A |      0...40 V           |          10 mV       |\n"
	"|ES 31 K2x40 R0.6 P  B |      0...40 V           |          10 mV       |\n"
	"--------------------------------------------------------------------------------------------------\n\n"
	"Note:\n"
	"(1) Value must be smaller than the voltage limit.\n",
                
                                   
    "This control defines the upper soft limit for the current\n"
    "setpoint, thus enabling you to prevent the output current from\n"
    "being set in excess of a certain value.\n\n"
	"Default Value: 0.6\n\n"
	"---------------------------------------------------------------------------------------------------\n"
	"|        Module             ch |  Valid Range (A)  |  Resolution (mA) |\n"
	"---------------------------------------------------------------------------------------------------\n"
	"|ES 31 K7 R7 P           A |       0...7.0 A       |          1 mA       |\n"
	"|ES 31 K7 R7 P           B |            -                |            -            |\n"
	"|ES 32 K30 R4 P         A |            -                |            -            |\n"
	"|ES 32 K30 R4 P         B |       0...4.0 A       |          1 mA       |\n"
	"|ES 32 K80 R1.5 P      A |            -                |            -            |\n"
	"|ES 32 K80 R1.5 P      B |       0...1.5 A       |          1 mA       |\n"
	"|ES 31 K2x8 R3 P       A |       0...3.0 A       |          1 mA       |\n"
	"|ES 31 K2x8 R3 P       B |       0...3.0 A       |          1 mA       |\n"
	"|ES 31 K2x16 R1.5 P  A |       0...1.5 A       |          1 mA       |\n"
	"|ES 31 K2x16 R1.5 P  B |       0...1.5 A       |          1 mA       |\n"
	"|ES 31 K2x40 R0.6 P  A |       0...0.6 A       |          1 mA       |\n"
	"|ES 31 K2x40 R0.6 P  B |       0...0.6 A       |          1 mA       |\n"
	"------------------------------------------------------------------------------------------------\n\n"
	"Note:\n"
	"(1) Value must be larger than current setpoint.\n",
	

    "This control defines the upper soft limit for the voltage\n"
    "setpoint, thus enabling you to prevent the output voltage from\n"
    "being set in excess of a certain value.\n\n"
    "Default Value: 7.0\n\n"
    "Valid Range:\n"
	"-------------------------------------------------------------------------------------------------\n"
	"|           Module           ch |  Valid Range (V)  |  Resolution (mV) |\n"
	"-------------------------------------------------------------------------------------------------\n"
	"|ES 31 K7 R7 P           A |       0... 7 V           |          1 mV         |\n"
	"|ES 31 K7 R7 P           B |           -                 |              -            |\n"
	"|ES 32 K30 R4 P         A |           -                 |              -            |\n"
	"|ES 32 K30 R4 P         B |      0...30 V           |           1 mV        |\n"
	"|ES 32 K80 R1.5 P      A |           -                 |              -            |\n"
	"|ES 32 K80 R1.5 P      B |      0...80 V           |           1 mV        |\n"
	"|ES 31 K2x8 R3 P       A |       0... 8 V           |           1 mV        |\n"
	"|ES 31 K2x8 R3 P       B |       0... 8 V           |           1 mV        |\n"
	"|ES 31 K2x16 R1.5 P  A |      0...16 V           |           1 mV        |\n"
	"|ES 31 K2x16 R1.5 P  B |      0...16 V           |           1 mV        |\n"
	"|ES 31 K2x40 R0.6 P  A |      0...40 V           |           1 mV        |\n"
	"|ES 31 K2x40 R0.6 P  B |      0...40 V           |           1 mV        |\n"
	"--------------------------------------------------------------------------------------------------\n\n"
    "Note:\n"
    "(1) Value must be larger than voltage setpoint.", 
    
    
    "This control specifies channel. \n\n"
	"Valid Channels: 1 up to 8\n\n"
	"Default Value: 1\n"
};
                                    
static char backgroundPnlHelpStr[] = {"This is a simple demo program using\n"
                                    "the Gossen Metrawatt Konstanter MSP Series\n"
                                    "instrument driver provided in the package.\n"};
                           
static char initPnlHelpStr[] =     {"This panel performs the following initialization actions:\n\n"
                                    "- Opens a instrument session.\n\n"
                                    "- Performs an identification query on the Instrument.\n\n"
                                    "- Sends initialization commands to the instrument that set any\n"
                                    "necessary programmatic variables to the state necessary\n"
                                    "for the operation of the instrument driver."};

static char configPnlHelpStr[] = {"This panel configures the instrument."};

/*= Main Function ===========================================================*/

void main ()
{
    // Load panels.
    panelHandle[BCKGRND] = LoadPanel (panelHandle[BCKGRND], "gmmspu.uir", BCKGRND);
    panelHandle[CONFIG] = LoadPanel (panelHandle[BCKGRND], "gmmspu.uir", CONFIG); 
    panelHandle[INIT] = LoadPanel (panelHandle[BCKGRND], "gmmspu.uir", INIT);
    // Display panels.
    DisplayPanel (panelHandle[BCKGRND]);
    DisplayPanel (panelHandle[INIT]);
    SetSleepPolicy (VAL_SLEEP_MORE);
    RunUserInterface ();
}   

/*===========================================================================*/
/* Function: Initialize Instrument                                           */
/* Purpose:  This is a callback function of the Continue button on the       */
/*           Initialize panel. It initializes the instrument and switches to */
/*           the panel Configure.                                            */
/*===========================================================================*/
int CVICALLBACK initInstrument (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    ViStatus error;
    ViChar instrDescr[256];
    ViInt16  rst, id;
    ViChar addr[16];
    char errMsg[256], errBuffer[256];

    switch (event) {
        case EVENT_COMMIT:
            SetWaitCursor(1);
            GetCtrlVal (panelHandle[INIT], INIT_ADDRESS, addr);
            GetCtrlVal (panelHandle[INIT], INIT_ID, &id);
            GetCtrlVal (panelHandle[INIT], INIT_RST, &rst);
            Fmt (instrDescr, "%s", addr);
            if ((error = gmmsp_init (instrDescr, id, rst, &instrHandle)) < 0 ) {
                gmmsp_error_message(VI_NULL, error, errMsg);
                sprintf (errBuffer, "Initialization Error:\n\n%s\n\nCheck your "
                        "connections and make sure you have the right GPIB address.",
                         errMsg);
				MessagePopup ("ERROR!", errBuffer);
                SetWaitCursor(0);
                return(0);   
            }
            HidePanel (panelHandle[INIT]);
            DisplayPanel (panelHandle[CONFIG]);
            SetWaitCursor(0); 
            break;
        case EVENT_RIGHT_CLICK:
            MessagePopup ("Help","This button causes the instrument to be initialized.");
            break;
    }
    return 0;
}

/*===========================================================================*/
/* Function: Configure Instrument                                            */
/* Purpose:  This is a callback function of the Continue button on the       */
/*           Configure panel .  					                         */
/*===========================================================================*/
int CVICALLBACK config (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    ViInt32 	errorCode = 0;
    ViStatus 	error;
    char 		errMsg[256], errorMessage[256], errBuffer[256];
    double 		volt, curr, volt_lim, curr_lim, nom_volt, nom_curr;
    int			channel;

    switch (event) {
        case EVENT_COMMIT:
    	    GetCtrlVal (panelHandle[CONFIG], CONFIG_CHANNEL, &channel);
            GetCtrlVal (panelHandle[CONFIG], CONFIG_VOLT, &volt);
            GetCtrlVal (panelHandle[CONFIG], CONFIG_CURR, &curr);
            GetCtrlVal (panelHandle[CONFIG], CONFIG_VOLT_LIMIT, &volt_lim);
            GetCtrlVal (panelHandle[CONFIG], CONFIG_CURR_LIMIT, &curr_lim);
            SetWaitCursor(1);
            
            if ((volt_lim < volt)||(curr_lim < curr))
            	{
				MessagePopup ("Warning", "Value must be smaller than limit!");
				if (volt_lim < volt) 
					{
					volt = volt_lim;
					SetCtrlVal (panelHandle[CONFIG], CONFIG_VOLT, volt);
					
					}
				if (curr_lim < curr) 
					{
					curr = curr_lim;
					SetCtrlVal (panelHandle[CONFIG], CONFIG_CURR, curr);
					
					}
            	}
            	
            	
            error = gmmsp_AppExample (instrHandle, channel, volt_lim, curr_lim, volt, curr,
                                    &nom_volt, &nom_curr);
            SetWaitCursor(0);
            if (error < 0) {
                gmmsp_error_message (VI_NULL, error, errMsg);
                gmmsp_error_query (instrHandle, &errorCode, errorMessage);
                sprintf (errBuffer, "Error Message: %s \n\nInstrument Specific Error: %d: %s",
                    errMsg, errorCode, errorMessage);
                MessagePopup ("Error:", errBuffer);
            }
            else {
                SetCtrlVal (panelHandle[CONFIG], CONFIG_VOLT_NOM, nom_volt);
                SetCtrlVal (panelHandle[CONFIG], CONFIG_CURR_NOM, nom_curr);
            }
            break;
        case EVENT_RIGHT_CLICK:
            MessagePopup ("Help","This button will configure the instrument and take its settings.");
            break;
    }
    return 0;
}

/*===========================================================================*/
/* Function: Cancel                                                          */
/* Purpose:  Called by the init panel this function pops up a confirmation   */
/*           dialog box and then quits the user interface, if desired.       */
/*===========================================================================*/
int CVICALLBACK Cancel (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    SetWaitCursor(0);
    switch (event) {
        case EVENT_COMMIT:
            if ((ConfirmPopup ("Exit Application", 
                   "Are you sure you want to quit this application?")) == 1) {
                QuitUserInterface (0);
            }
            break;
        case EVENT_RIGHT_CLICK:
            MessagePopup ("Control Help", "Closes the application.");
            break;
    }
    return 0;
}

/*===========================================================================*/
/* Function: Control Help                                                    */
/* Purpose:  This is a callback function of all controls that configure the  */
/*           instrument. On the right mouse-click on the control a help      */
/*           window describing its purpose is displayed.                     */
/*===========================================================================*/
int CVICALLBACK controlHelp (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{   
    SetWaitCursor(0);
    if (event == EVENT_RIGHT_CLICK) {
        if (panel == panelHandle[INIT]) 
            MessagePopup ("Help", initCtrlsHelpStr[control-4]);
        if (panel == panelHandle[CONFIG]) 
            MessagePopup ("Help", configCtrlsHelpStr[control-4]);
    }
    return 0;
}

/*===========================================================================*/
/* Function: Panel Help                                                      */
/* Purpose:  This is a callback function of the menu bar. It displays a help */
/*           window describing panel being used.                             */
/*===========================================================================*/
void CVICALLBACK panelHelp (int menuBar, int menuItem, void *callbackData,
        int panel)
{
    SetWaitCursor(0);
    if (panel == panelHandle[BCKGRND])
        MessagePopup ("Help", backgroundPnlHelpStr);
    if (panel == panelHandle[INIT])
        MessagePopup ("Help", initPnlHelpStr);
    if (panel == panelHandle[CONFIG])
        MessagePopup ("Help", configPnlHelpStr);
    return;
}

/*===========================================================================*/
/* Function: Launch Close Panel                                              */
/* Purpose:  This is a callback function of the button Close on the          */
/*           panel Configure. It pops-up a panel that will close the         */
/*           instrument.                                                     */
/*===========================================================================*/
int CVICALLBACK launchClose (int panel, int control, int event,
        void *callbackData, int eventData1, int eventData2)
{
    
    SetWaitCursor(0);
    switch (event) {
        case EVENT_COMMIT:                                       
            if ((ConfirmPopup ("Exit Application", "Are you sure you want to quit "
                    "this application?")) == 1) 
            {
                gmmsp_close (instrHandle);
                QuitUserInterface (0);
                exit (-1);
            }
            break;
        case EVENT_RIGHT_CLICK:
            MessagePopup ("Help","This button will pop-up a panel to close the instrument.");
            break;
    }
    return 0;
}   

/*= End =====================================================================*/
